//###########################################################################
// This file is part of LImA, a Library for Image Acquisition
//
// Copyright (C) : 2009-2011
// European Synchrotron Radiation Facility
// BP 220, Grenoble 38043
// FRANCE
//
// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
//###########################################################################
#ifndef CTACQUISITION_H
#define CTACQUISITION_H

#include <algorithm>
#include <sstream>

#include "lima/LimaCompatibility.h"
#include "lima/Constants.h"
#include "lima/HwInterface.h"
#include "lima/HwCap.h"
#include "lima/CtControl.h"
#include "lima/CtConfig.h"
#include "lima/Debug.h"

namespace lima
{
  /// This class control the acquisition of images given a hardware
  /// interface.
  class LIMACORE_API CtAcquisition
  {
    DEB_CLASS_NAMESPC(DebModControl,"Acquisition","Control");
    friend class CtControl;

  public:
    enum AccTimeMode {Live,Real};
    enum AutoExposureMode {
      OFF,	  ///< Always off
      ON,	  ///< Always on
      ON_LIVE	  ///< ON during live and OFF for standard Acquisition
    };
    typedef std::vector<AutoExposureMode> AutoExposureModeList;

    struct LIMACORE_API Parameters {
      DEB_CLASS_NAMESPC(DebModControl,"Acquisition::Parameters","Control");
    public:
      Parameters();
      void reset();
      
      AcqMode		acqMode;
      AccTimeMode 	accTimeMode;
      int		acqNbFrames;
      double		acqExpoTime;
      double		accMaxExpoTime;
      int		concatNbFrames;
      double		latencyTime;
      TrigMode 		triggerMode;
      AutoExposureMode  autoExpoMode;
    };

    CtAcquisition(HwInterface *hw);
    ~CtAcquisition();

    // --- global

    void setPars(const Parameters &pars);
    void getPars(Parameters& pars) const;

    void reset();
    void apply(CtControl::ApplyPolicy policy, CtControl *control=NULL);
    void sync();

    // --- acq modes

    void setAcqMode(AcqMode mode);
    void getAcqMode(AcqMode& mode) const;

    void setAccTimeMode(AccTimeMode mode);
    void getAccTimeMode(AccTimeMode &mode) const;

    void setAcqNbFrames(int nframes);
    void getAcqNbFrames(int& nframes) const;

    void setAcqExpoTime(double acq_time);
    void getAcqExpoTime(double& acq_time) const;
    bool checkAutoExposureMode(AutoExposureMode mode) const;
    void getAutoExposureModeList(AutoExposureModeList& modes) const;
    void setAutoExposureMode(AutoExposureMode mode);
    void getAutoExposureMode(AutoExposureMode& mode) const;


    void setAccMaxExpoTime(double max_time);
    void getAccMaxExpoTime(double& max_time) const;

    void getAccNbFrames(int& nframes) const;
    void getAccExpoTime(double& acc_time) const;
    void getAccLiveTime(double& acc_live_time) const;
    void getAccDeadTime(double& acc_dead_time) const;

    void setConcatNbFrames(int nframes);
    void getConcatNbFrames(int& nframes) const; 

    // --- common

    void setLatencyTime(double latency_time);
    void getLatencyTime(double& latency_time) const;

    void setTriggerMode(TrigMode mode);
    void getTriggerMode(TrigMode& mode) const;

    void getTriggerModeList(TrigModeList& modes) const;
    
    bool isMonitorMode() const {return m_monitor_mode;}
  private:
    class _ValidRangesCallback;
    friend class _ValidRangesCallback;

    struct ChangedPars {
      DEB_CLASS_NAMESPC(DebModControl,"Acquisition::ChangedPars","Control");
    public:
      ChangedPars();
      void set(bool);
      void check(Parameters p1, Parameters p2);

      bool	acqExpoTime;
      bool	acqNbFrames;
      bool	latencyTime;
      bool	triggerMode;
      bool	accMaxExpoTime;
      bool	acqMode;
      bool      autoExpoMode;
    };

    void _updateAccPars() const;
    void _setDefaultPars(Parameters* pars);
    void _apply();
    void _hwRead();
    void _check_timing_ranges();
#ifdef WITH_CONFIG
    class _ConfigHandler;
    CtConfig::ModuleTypeCallback* _getConfigHandler();
#endif //WITH_CONFIG

    HwSyncCtrlObj	*m_hw_sync;
    HwSyncCtrlObj::ValidRangesType	m_valid_ranges;
    Parameters	m_inpars, m_hwpars;
    ChangedPars	m_changes;
    double		m_readout_time;
    double		m_frame_rate;
    mutable int		m_acc_nframes;
    mutable double	m_acc_exptime;
    mutable double	m_acc_live_time;
    mutable double	m_acc_dead_time;
    bool		m_applied_once;
    _ValidRangesCallback *m_valid_ranges_cb;
    bool		m_monitor_mode;
  };

  inline const char* convert_2_string(CtAcquisition::AccTimeMode accTimeMode)
    {
      const char *name = "Unknown";
      switch(accTimeMode)
	{
	case CtAcquisition::Live: name = "Live"; break;
	case CtAcquisition::Real: name = "Real"; break;
	}
      return name;
    }
  inline void convert_from_string(const std::string& val,
				  CtAcquisition::AccTimeMode& accTimeMode)
    {
      std::string buffer = val;
      std::transform(buffer.begin(),buffer.end(),
		     buffer.begin(),::tolower);
      if(buffer == "live")
	accTimeMode = CtAcquisition::Live;
      else if(buffer == "real")
	accTimeMode = CtAcquisition::Real;
      else
	{
	  std::ostringstream msg;
	  msg << "AccTimeMode can't be:" << DEB_VAR1(val);
	  throw LIMA_EXC(Control,InvalidValue,msg.str());
	}
    }
  inline std::ostream& operator<<(std::ostream& os,
				  const CtAcquisition::AccTimeMode& mode)
  {
    return os << convert_2_string(mode);
  }
  inline std::istream& operator>>(std::istream& is,
				  CtAcquisition::AccTimeMode& mode)
  {
    std::string s;
    is >> s;
    convert_from_string(s, mode);
    return is;
  }
  inline std::ostream& operator<<(std::ostream &os,const CtAcquisition::Parameters &params)
  {
    os << "<"
       << "acqMode=" << params.acqMode << ", "
       << "acqNbFrames=" << params.acqNbFrames << ", "
       << "acqExpoTime=" << params.acqExpoTime << ", "
       << "accMaxExpoTime=" << params.accMaxExpoTime << ", "
       << "concatNbFrames=" << params.concatNbFrames << ", "
       << "latencyTime=" << params.latencyTime << ", "
       << "triggerMode=" << params.triggerMode << ", "
       << "autoExpoMode=" << params.autoExpoMode
       << ">";
    return os; 
  }

  inline const char* convert_2_string(CtAcquisition::AutoExposureMode mode)
  {
    const char* aHumanReadablePt;
    switch(mode)
      {
      case CtAcquisition::OFF:		aHumanReadablePt = "OFF";	break;
      case CtAcquisition::ON:		aHumanReadablePt = "ON";	break;
      case CtAcquisition::ON_LIVE:	aHumanReadablePt = "ON LIVE";	break;
      default:
	aHumanReadablePt = "UNKNOWN";
	break;
      }
    return aHumanReadablePt;
  }

  inline void convert_from_string(const std::string& val,
				  CtAcquisition::AutoExposureMode& mode)
  {
    std::string buffer = val;
    std::transform(buffer.begin(),buffer.end(),
		   buffer.begin(),::tolower);
    
    if(buffer == "off") mode = CtAcquisition::OFF;
    else if(buffer == "on") mode = CtAcquisition::ON;
    else if(buffer == "on live") mode = CtAcquisition::ON_LIVE;
    else
      {
	std::ostringstream msg;
	msg << "AutoExposureMode can't be:" << DEB_VAR1(val);
	throw LIMA_EXC(Common,InvalidValue,msg.str());
      }
  }
  inline std::ostream& operator<<(std::ostream& os,
				  const CtAcquisition::AutoExposureMode& mode)
  {
    return os << convert_2_string(mode);
  }
  inline std::istream& operator>>(std::istream& is,
				  CtAcquisition::AutoExposureMode& mode)
  {
    std::string s;
    is >> s;
    convert_from_string(s, mode);
    return is;
  }

  
} // namespace lima

#endif // CTACQUISITION_H
